CREATE TABLE `_version` (
`version` VARCHAR(50),
PRIMARY KEY ( `version`)) ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE category (
  category_id       INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
  long_description  TEXT   NULL,
  image_path        VARCHAR(255)   NULL,
  asset_flag        BIT   NOT NULL   COMMENT 'bit specifying whether or not this category applies to assets',
  inventory_flag    BIT   NOT NULL   COMMENT 'bit specifying whether or not this category applies to inventory',
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( category_id ),
    INDEX category_fkindex1 ( created_by ),
    INDEX category_fkindex2 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE manufacturer (
  manufacturer_id   INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
  long_description  TEXT   NULL,
  image_path        VARCHAR(255)   NULL,
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( manufacturer_id ),
    INDEX manufacturer_fkindex1 ( created_by ),
    INDEX manufacturer_fkindex2 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE location (
  location_id       INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
  long_description  TEXT   NULL,
  enabled_flag      BIT (1)   NOT NULL   DEFAULT b'1',
  asset_flag        BIT (1)   NOT NULL   DEFAULT b'1',
  inventory_flag    BIT (1)   NOT NULL   DEFAULT b'1',
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( location_id ),
    INDEX location_fkindex1 ( created_by ),
    INDEX location_fkindex2 ( modified_by ),
    UNIQUE (short_description ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE asset_model (
  asset_model_id    INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  category_id       INTEGER UNSIGNED   NULL,
  manufacturer_id   INTEGER UNSIGNED   NULL,
  asset_model_code  VARCHAR(50)   NULL,
  short_description VARCHAR(255)   NOT NULL,
  long_description  TEXT   NULL,
  image_path        VARCHAR(255)   NULL,
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
  `depreciation_class_id` INTEGER UNSIGNED   NULL,
    PRIMARY KEY ( asset_model_id ),
    INDEX asset_model_fkindex1 ( category_id ),
    INDEX asset_model_fkindex2 ( manufacturer_id ),
    INDEX asset_model_fkindex3 ( created_by ),
    INDEX asset_model_fkindex4 ( modified_by ),
    INDEX asset_model_fkindex5 ( depreciation_class_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE `asset` (
  `asset_id`         INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  `parent_asset_id` INTEGER UNSIGNED DEFAULT NULL,
  `asset_model_id`   INTEGER UNSIGNED   NOT NULL,
  `location_id`      INTEGER UNSIGNED   NULL,
  `asset_code`       VARCHAR(50)   NOT NULL,
  `image_path`       VARCHAR(255)   NULL,
  `checked_out_flag` BIT   NULL,
  `reserved_flag`    BIT   NULL,
  `linked_flag`		 BIT(1) DEFAULT NULL,
  `archived_flag`    BIT(1) DEFAULT NULL,
  `created_by`       INTEGER UNSIGNED   NULL,
  `creation_date`    DATETIME   NULL   DEFAULT NULL,
  `modified_by`      INTEGER UNSIGNED   NULL,
  `modified_date`    TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
  `depreciation_flag` BIT(1) DEFAULT NULL,
  `purchase_date`    DATE  DEFAULT NULL,
  `purchase_cost`    DECIMAL(10,2) DEFAULT NULL,
    PRIMARY KEY ( `asset_id` ),
    INDEX asset_fkindex1 ( `asset_model_id` ),
    INDEX asset_fkindex2 ( `location_id` ),
    INDEX asset_fkindex3 ( `created_by` ),
    INDEX asset_fkindex4 ( `modified_by` ),
    INDEX asset_fkindex5 ( `parent_asset_id` ),
    INDEX `parent_asset_id_linked` ( `parent_asset_id` , `linked_flag` ),
    UNIQUE (asset_code ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE asset_transaction (
  asset_transaction_id        INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  asset_id                    INTEGER UNSIGNED   NOT NULL,
  transaction_id              INTEGER UNSIGNED   NOT NULL,
  parent_asset_transaction_id INTEGER UNSIGNED   NULL,
  source_location_id          INTEGER UNSIGNED   NULL,
  destination_location_id     INTEGER UNSIGNED   NULL,
  new_asset_flag              BIT   NULL   COMMENT 'Set to true if a new asset was created while creating the transaction (a receipt, or shipment, e.g.)',
  new_asset_id                INTEGER UNSIGNED   NULL,
  schedule_receipt_flag       BIT   NULL   COMMENT 'Set to true if an asset is to be scheduled for a receipt',
  schedule_receipt_due_date   DATE   NULL   COMMENT 'Placeholder for automatically scheduled receipts until the shipment is completed',
  created_by                  INTEGER UNSIGNED   NULL,
  creation_date               DATETIME   NULL   DEFAULT NULL,
  modified_by                 INTEGER UNSIGNED   NULL,
  modified_date               TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( asset_transaction_id ),
    INDEX asset_transaction_fkindex2 ( transaction_id ),
    INDEX asset_transaction_fkindex1 ( asset_id ),
    INDEX asset_transaction_fkindex3 ( source_location_id ),
    INDEX asset_transaction_fkindex4 ( destination_location_id ),
    INDEX asset_transaction_fkindex5 ( created_by ),
    INDEX asset_transaction_fkindex6 ( modified_by ),
    INDEX asset_transaction_fkindex7 ( new_asset_id ),
    INDEX asset_transaction_fkindex8 ( parent_asset_transaction_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE transaction_type (
  transaction_type_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description   VARCHAR(50)   NOT NULL,
  asset_flag          BIT   NOT NULL   DEFAULT 0,
  inventory_flag      BIT   NOT NULL   DEFAULT 0,
    PRIMARY KEY ( transaction_type_id ),
    UNIQUE (short_description ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE inventory_model (
  inventory_model_id   INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  category_id          INTEGER UNSIGNED   NULL,
  manufacturer_id      INTEGER UNSIGNED   NULL,
  inventory_model_code VARCHAR(50)   NOT NULL   COMMENT 'Bar code',
  short_description    VARCHAR(255)   NOT NULL,
  long_description     TEXT   NULL,
  image_path           VARCHAR(255)   NULL,
  price                DECIMAL(12,2)   NULL,
  created_by           INTEGER UNSIGNED   NULL,
  creation_date        DATETIME   NULL   DEFAULT NULL,
  modified_by          INTEGER UNSIGNED   NULL,
  modified_date        TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( inventory_model_id ),
    INDEX inventory_model_fkindex1 ( category_id ),
    INDEX inventory_model_fkindex2 ( manufacturer_id ),
    INDEX inventory_model_fkindex3 ( created_by ),
    INDEX inventory_model_fkindex4 ( modified_by ),
    UNIQUE (inventory_model_code ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE inventory_location (
  inventory_location_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  inventory_model_id    INTEGER UNSIGNED   NOT NULL,
  location_id           INTEGER UNSIGNED   NOT NULL,
  quantity              INTEGER UNSIGNED   NOT NULL,
  created_by            INTEGER UNSIGNED   NULL,
  creation_date         DATETIME   NULL   DEFAULT NULL,
  modified_by           INTEGER UNSIGNED   NULL,
  modified_date         TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( inventory_location_id ),
    INDEX inventory_location_fkindex1 ( location_id ),
    INDEX inventory_location_fkindex2 ( inventory_model_id ),
    INDEX inventory_location_fkindex3 ( modified_by ),
    INDEX inventory_location_fkindex4 ( created_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE inventory_transaction (
  inventory_transaction_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  inventory_location_id    INTEGER UNSIGNED   NOT NULL,
  transaction_id           INTEGER UNSIGNED   NOT NULL,
  quantity                 INTEGER UNSIGNED   NOT NULL,
  source_location_id       INTEGER UNSIGNED   NULL,
  destination_location_id  INTEGER UNSIGNED   NULL,
  created_by               INTEGER UNSIGNED   NULL,
  creation_date            DATETIME   NULL   DEFAULT NULL,
  modified_by              INTEGER UNSIGNED   NULL,
  modified_date            TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( inventory_transaction_id ),
    INDEX inventory_transaction_fkindex2 ( transaction_id ),
    INDEX inventory_transaction_fkindex3 ( source_location_id ),
    INDEX inventory_transaction_fkindex4 ( destination_location_id ),
    INDEX inventory_transaction_fkindex1 ( inventory_location_id ),
    INDEX inventory_transaction_fkindex5 ( created_by ),
    INDEX inventory_transaction_fkindex6 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE user_account (
  user_account_id       INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  first_name            VARCHAR(50)   NOT NULL,
  last_name             VARCHAR(50)   NOT NULL,
  username              VARCHAR(30)   NOT NULL,
  password_hash         VARCHAR(60)   NOT NULL,
  email_address         VARCHAR(128)  NULL,
  active_flag           BIT   NOT NULL   COMMENT 'User account enabled/disabled',
  admin_flag            BIT   NOT NULL   COMMENT 'Designates user as normal or administrator',
  owner_flag            BIT   NOT NULL  DEFAULT 0,
  portable_access_flag  BIT   NULL,
  portable_user_pin     INT(10)   NULL,
  role_id               INTEGER UNSIGNED   NOT NULL,
  password_reset_code   VARCHAR(64) DEFAULT NULL,
  password_reset_expiry DATETIME DEFAULT NULL,
  created_by            INTEGER UNSIGNED   NULL,
  creation_date         DATETIME   NULL   DEFAULT NULL,
  modified_by           INTEGER UNSIGNED   NULL,
  modified_date         TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( user_account_id ),
    INDEX user_account_fkindex1 ( created_by ),
    INDEX user_account_fkindex2 ( modified_by ),
    INDEX user_account_fkindex3 ( role_id ),
    UNIQUE (username ),
    UNIQUE (email_address))
COMMENT 'User accounts are stored in this table'
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE transaction (
  transaction_id      INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  entity_qtype_id     INTEGER UNSIGNED   NOT NULL,
  transaction_type_id INTEGER UNSIGNED   NOT NULL,
  note                TEXT   NULL,
  created_by          INTEGER UNSIGNED   NULL,
  creation_date       DATETIME   NULL   DEFAULT NULL,
  modified_by         INTEGER UNSIGNED   NULL,
  modified_date       TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( transaction_id ),
    INDEX transaction_fkindex1 ( transaction_type_id ),
    INDEX transaction_fkindex2 ( created_by ),
    INDEX transaction_fkindex3 ( modified_by ),
    INDEX transaction_fkindex4 ( entity_qtype_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE custom_field (
  custom_field_id               INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  custom_field_qtype_id         INTEGER UNSIGNED   NOT NULL,
  default_custom_field_value_id INTEGER UNSIGNED   NULL,
  short_description             VARCHAR(255)   NOT NULL,
  active_flag                   BIT   NULL,
  required_flag                 BIT   NULL,
  all_asset_models_flag         BIT   NULL  DEFAULT 1,
  searchable_flag               BIT   NULL,
  created_by                    INTEGER UNSIGNED   NULL,
  creation_date                 DATETIME   NULL,
  modified_by                   INTEGER UNSIGNED   NULL,
  modified_date                 TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( custom_field_id ),
    INDEX custom_field_fkindex2 ( modified_by ),
    INDEX custom_field_fkindex3 ( created_by ),
    INDEX custom_field_fkindex1 ( custom_field_qtype_id ),
    INDEX custom_field_fkindex4 ( default_custom_field_value_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE custom_field_value (
  custom_field_value_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  custom_field_id       INTEGER UNSIGNED   NOT NULL,
  short_description     TEXT   NULL,
  created_by            INTEGER UNSIGNED   NULL,
  creation_date         DATETIME   NULL,
  modified_by           INTEGER UNSIGNED   NULL,
  modified_date         TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( custom_field_value_id ),
    INDEX custom_field_value_fkindex2 ( created_by ),
    INDEX custom_field_value_fkindex3 ( modified_by ),
    INDEX custom_field_value_fkindex1 ( custom_field_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE custom_field_qtype (
  custom_field_qtype_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  name                  VARCHAR(10)   NOT NULL,
    PRIMARY KEY ( custom_field_qtype_id ),
    UNIQUE (name ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE entity_qtype (
  entity_qtype_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  name            VARCHAR(50)   NOT NULL,
    PRIMARY KEY ( entity_qtype_id ),
    UNIQUE (name ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE entity_qtype_custom_field (
  entity_qtype_custom_field_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  entity_qtype_id              INTEGER UNSIGNED   NOT NULL,
  custom_field_id              INTEGER UNSIGNED   NOT NULL,
    PRIMARY KEY ( entity_qtype_custom_field_id ),
    INDEX entity_qtype_custom_field_fkindex1 ( entity_qtype_id ),
    INDEX entity_qtype_custom_field_fkindex2 ( custom_field_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE asset_custom_field_asset_model (
  asset_custom_field_asset_model_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  asset_model_id              INTEGER UNSIGNED   NOT NULL,
  custom_field_id             INTEGER UNSIGNED   NOT NULL,
    PRIMARY KEY ( asset_custom_field_asset_model_id ),
    INDEX asset_custom_field_asset_model_fkindex1 ( asset_model_id ),
    INDEX asset_custom_field_asset_model_fkindex2 ( custom_field_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE company (
  company_id        INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  address_id        INTEGER UNSIGNED   NULL,
  short_description VARCHAR(255)   NOT NULL,
  website           VARCHAR(255)   NULL,
  telephone         VARCHAR(50)   NULL,
  fax               VARCHAR(50)   NULL,
  email             VARCHAR(50)   NULL,
  long_description  TEXT   NULL,
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( company_id ),
    INDEX company_fkindex1 ( address_id ),
    INDEX company_fkindex2 ( created_by ),
    INDEX company_fkindex3 ( modified_by ),
    UNIQUE (short_description ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE contact (
  contact_id    INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  company_id    INTEGER UNSIGNED   NOT NULL,
  address_id    INTEGER UNSIGNED   NULL,
  first_name    VARCHAR(50)   NULL,
  last_name     VARCHAR(50)   NOT NULL,
  title         VARCHAR(50)   NULL,
  email         VARCHAR(50)   NULL,
  phone_office  VARCHAR(50)   NULL,
  phone_home    VARCHAR(50)   NULL,
  phone_mobile  VARCHAR(50)   NULL,
  fax           VARCHAR(50)   NULL,
  description   TEXT   NULL,
  created_by    INTEGER UNSIGNED   NULL,
  creation_date DATETIME   NULL   DEFAULT NULL,
  modified_by   INTEGER UNSIGNED   NULL,
  modified_date TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( contact_id ),
    INDEX contact_fkindex3 ( modified_by ),
    INDEX contact_fkindex4 ( created_by ),
    INDEX contact_fkindex2 ( address_id ),
    INDEX contact_fkindex1 ( company_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE address (
  address_id        INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  company_id        INTEGER UNSIGNED   NOT NULL,
  short_description VARCHAR(255)   NOT NULL,
  country_id        INTEGER UNSIGNED   NOT NULL,
  address_1         VARCHAR(255)   NOT NULL,
  address_2         VARCHAR(255)   NULL,
  city              VARCHAR(50)   NOT NULL,
  state_province_id INTEGER UNSIGNED   NULL,
  postal_code       VARCHAR(50)   NOT NULL,
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( address_id ),
    INDEX address_fkindex1 ( company_id ),
    INDEX address_fkindex2 ( country_id ),
    INDEX address_fkindex3 ( state_province_id ),
    INDEX address_fkindex4 ( modified_by ),
    INDEX address_fkindex5 ( created_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE country (
  country_id        INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(50)   NOT NULL,
  abbreviation      CHAR(2)   NULL,
  state_flag        BIT   NULL,
  province_flag     BIT   NULL,
    PRIMARY KEY ( country_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE state_province (
  state_province_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  country_id        INTEGER UNSIGNED   NULL,
  short_description VARCHAR(50)   NULL,
  abbreviation      VARCHAR(2)   NULL,
    PRIMARY KEY ( state_province_id ),
    INDEX state_province_fkindex1 ( country_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE shipment (
  shipment_id     INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  shipment_number VARCHAR(50)   NOT NULL,
  transaction_id  INTEGER UNSIGNED   NOT NULL,
  from_company_id INTEGER UNSIGNED   NOT NULL,
  from_contact_id INTEGER UNSIGNED   NOT NULL,
  from_address_id INTEGER UNSIGNED   NOT NULL,
  to_company_id   INTEGER UNSIGNED   NOT NULL,
  to_contact_id   INTEGER UNSIGNED   NOT NULL,
  to_address_id   INTEGER UNSIGNED   NOT NULL,
  courier_id      INTEGER UNSIGNED   NULL,
  tracking_number VARCHAR(50)   NULL,
  ship_date       DATE   NOT NULL,
  shipped_flag    BIT   NULL,
  created_by      INTEGER UNSIGNED   NULL,
  creation_date   DATETIME   NULL,
  modified_by     INTEGER UNSIGNED   NULL,
  modified_date   TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( shipment_id ),
    INDEX shipment_fkindex1 ( transaction_id ),
    INDEX shipment_fkindex4 ( from_address_id ),
    INDEX shipment_fkindex5 ( to_address_id ),
    INDEX shipment_fkindex6 ( to_company_id ),
    INDEX shipment_fkindex8 ( courier_id ),
    INDEX shipment_fkindex13 ( created_by ),
    INDEX shipment_fkindex14 ( modified_by ),
    INDEX shipment_fkindex2 ( from_contact_id ),
    INDEX shipment_fkindex3 ( to_contact_id ),
    UNIQUE (shipment_number ),
    UNIQUE (transaction_id ),
    INDEX shipment_fkindex16 ( from_company_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE courier (
  courier_id        INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
  active_flag       BIT   NULL,
    PRIMARY KEY ( courier_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE receipt (
  receipt_id      INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  transaction_id  INTEGER UNSIGNED   NOT NULL,
  from_company_id INTEGER UNSIGNED   NOT NULL,
  from_contact_id INTEGER UNSIGNED   NOT NULL,
  to_contact_id   INTEGER UNSIGNED   NOT NULL,
  to_address_id   INTEGER UNSIGNED   NOT NULL,
  receipt_number  VARCHAR(50)   NOT NULL,
  due_date        DATE   NULL,
  receipt_date    DATE   NULL,
  received_flag   BIT   NULL,
  created_by      INTEGER UNSIGNED   NULL,
  creation_date   DATETIME   NULL,
  modified_by     INTEGER UNSIGNED   NULL,
  modified_date   TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( receipt_id ),
    INDEX receipt_fkindex1 ( from_company_id ),
    INDEX receipt_fkindex2 ( from_contact_id ),
    INDEX receipt_fkindex3 ( to_contact_id ),
    INDEX receipt_fkindex4 ( to_address_id ),
    INDEX receipt_fkindex5 ( created_by ),
    INDEX receipt_fkindex6 ( modified_by ),
    INDEX receipt_fkindex7 ( transaction_id ),
    UNIQUE (transaction_id ),
    INDEX receipt_index3241 ( receipt_number ),
    UNIQUE (receipt_number ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE role (
  role_id           INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
  long_description  TEXT   NULL,
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( role_id ),
    INDEX role_fkindex1 ( created_by ),
    INDEX role_fkindex2 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE module (
  module_id         INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NULL,
    PRIMARY KEY ( module_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE role_module (
  role_module_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  role_id        INTEGER UNSIGNED   NOT NULL,
  module_id      INTEGER UNSIGNED   NOT NULL,
  access_flag    BIT   NOT NULL   COMMENT 'Determines whether this role has access to this module',
  created_by     INTEGER UNSIGNED   NULL,
  creation_date  DATETIME   NULL   DEFAULT NULL,
  modified_by    INTEGER UNSIGNED   NULL,
  modified_date  TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( role_module_id ),
    INDEX role_module_fkindex1 ( role_id ),
    INDEX role_module_fkindex2 ( module_id ),
    INDEX role_module_fkindex3 ( created_by ),
    INDEX role_module_fkindex4 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE authorization (
  authorization_id  INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NULL,
    PRIMARY KEY ( authorization_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE authorization_level (
  authorization_level_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description      VARCHAR(255)   NULL,
    PRIMARY KEY ( authorization_level_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE role_module_authorization (
  role_module_authorization_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  role_module_id               INTEGER UNSIGNED   NULL,
  authorization_id             INTEGER UNSIGNED   NULL,
  authorization_level_id       INTEGER UNSIGNED   NULL,
  created_by                   INTEGER UNSIGNED   NULL,
  creation_date                DATETIME   NULL   DEFAULT NULL,
  modified_by                  INTEGER UNSIGNED   NULL,
  modified_date                TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( role_module_authorization_id ),
    INDEX role_module_authorization_fkindex1 ( role_module_id ),
    INDEX role_module_authorization_fkindex2 ( authorization_id ),
    INDEX role_module_authorization_fkindex3 ( authorization_level_id ),
    INDEX role_module_authorization_fkindex4 ( created_by ),
    INDEX role_module_authorization_fkindex5 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE role_transaction_type_authorization (
  role_transaction_type_authorization_id INT(10) NOT NULL AUTO_INCREMENT,
  role_id INTEGER UNSIGNED NOT NULL,
  transaction_type_id INTEGER UNSIGNED NOT NULL,
  authorization_level_id INTEGER UNSIGNED NOT NULL,
  created_by INTEGER UNSIGNED NULL,
  creation_date DATETIME NULL DEFAULT NULL,
  modified_by INTEGER UNSIGNED NULL,
  modified_date TIMESTAMP ON UPDATE CURRENT_TIMESTAMP NULL DEFAULT NULL,
    PRIMARY KEY(role_transaction_type_authorization_id),
    INDEX role_transaction_type_authorization_FKIndex1(created_by),
    INDEX role_transaction_type_authorization_FKIndex2(modified_by),
    INDEX role_transaction_type_authorization_FKIndex3(authorization_level_id),
    INDEX role_transaction_type_authorization_FKIndex4(transaction_type_id),
    INDEX role_transaction_type_authorization_FKIndex5(role_id),
    UNIQUE role_transaction_type_authorization_UNIQUE(role_id, transaction_type_id)
)
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE `admin_setting` (
  `setting_id`        INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  `short_description` VARCHAR(255)   NOT NULL,
  `value`             TEXT   NULL,
    PRIMARY KEY ( `setting_id` ),
    UNIQUE (`short_description` ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE shortcut (
  shortcut_id       INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  module_id         INTEGER UNSIGNED   NOT NULL,
  authorization_id  INTEGER UNSIGNED   NULL,
  transaction_type_id INTEGER UNSIGNED NULL,
  short_description VARCHAR(255)   NOT NULL,
  link              VARCHAR(255)   NOT NULL,
  image_path        VARCHAR(255)   NULL,
  entity_qtype_id   INTEGER UNSIGNED   NOT NULL,
  create_flag       TINYINT(1) UNSIGNED   NOT NULL,
    PRIMARY KEY ( shortcut_id ),
    INDEX shortcut_fkindex1 ( module_id ),
    INDEX shortcut_fkindex2 ( authorization_id ),
    INDEX shortcut_fkindex3 ( transaction_type_id ),
    INDEX shortcut_fkindex4 ( entity_qtype_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE datagrid (
  datagrid_id       INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
    PRIMARY KEY ( datagrid_id ),
    UNIQUE (short_description ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE datagrid_column_preference (
  datagrid_column_preference_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  datagrid_id                   INTEGER UNSIGNED   NOT NULL,
  column_name                   VARCHAR(255)   NOT NULL,
  user_account_id               INTEGER UNSIGNED   NOT NULL,
  display_flag                  BIT   NOT NULL   DEFAULT 1,
    PRIMARY KEY ( datagrid_column_preference_id ),
    UNIQUE (datagrid_id,column_name,user_account_id ),
    INDEX datagrid_column_preference_fkindex1 ( datagrid_id ),
    INDEX datagrid_column_preference_fkindex2 ( user_account_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE notification (
  notification_id   INT(10)   NOT NULL   AUTO_INCREMENT,
  short_description VARCHAR(255)   NOT NULL,
  long_description  TEXT   NULL,
  criteria          VARCHAR(255)   NULL   DEFAULT NULL,
  frequency         ENUM('once','daily','weekly','monthly')   NOT NULL,
  enabled_flag      BIT(1)   NOT NULL   DEFAULT false,
  created_by        INTEGER UNSIGNED   NULL,
  creation_date     DATETIME   NULL   DEFAULT NULL,
  modified_by       INTEGER UNSIGNED   NULL,
  modified_date     TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( notification_id ),
    INDEX notification_fkindex1 ( created_by ),
    INDEX notification_fkindex2 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE notification_user_account (
  notification_user_account_id INT(10)   NOT NULL   AUTO_INCREMENT,
  user_account_id              INTEGER UNSIGNED   NOT NULL,
  notification_id              INT(10)   NOT NULL,
  level                        ENUM('all','owner')   NULL,
    PRIMARY KEY ( notification_user_account_id ),
    INDEX notification_user_account_fkindex1 ( notification_id ),
    INDEX notification_user_account_fkindex2 ( user_account_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE attachment (
  attachment_id   INT(10)   NOT NULL   AUTO_INCREMENT,
  entity_qtype_id INTEGER UNSIGNED   NOT NULL,
  entity_id       INTEGER UNSIGNED   NOT NULL,
  filename        VARCHAR(255)   NOT NULL,
  tmp_filename    VARCHAR(40)   NULL,
  file_type       VARCHAR(40)   NULL,
  path            VARCHAR(255)   NULL,
  SIZE            INT(10)   NULL,
  created_by      INTEGER UNSIGNED   NOT NULL,
  creation_date   DATETIME   NOT NULL,
    PRIMARY KEY ( attachment_id ),
    INDEX attachment_fkindex1 ( entity_qtype_id ),
    INDEX ( entity_id ),
    INDEX attachment_fkindex2 ( created_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE audit (
  audit_id        INT(10)   NOT NULL   AUTO_INCREMENT   COMMENT 'PK',
  entity_qtype_id INTEGER UNSIGNED   NOT NULL,
  created_by      INTEGER UNSIGNED   NULL,
  creation_date   DATETIME   NULL,
  modified_by     INTEGER UNSIGNED   NULL,
  modified_date   TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL,
    PRIMARY KEY ( audit_id ),
    INDEX audit_fkindex1 ( entity_qtype_id ),
    INDEX audit_fkindex2 ( created_by ),
    INDEX audit_fkindex3 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE audit_scan (
  audit_scan_id INT(10)   NOT NULL   AUTO_INCREMENT   COMMENT 'PK',
  audit_id      INT(10)   NOT NULL,
  location_id   INTEGER UNSIGNED   NOT NULL,
  entity_id     INT(10)   NULL   COMMENT 'FK',
  COUNT         INT(10)   NULL,
  system_count  INT(10)   NULL,
    PRIMARY KEY ( audit_scan_id ),
    INDEX audit_scan_fkindex1 ( audit_id ),
    INDEX audit_scan_fkindex2 ( location_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE role_entity_qtype_built_in_authorization (
  role_entity_built_in_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  role_id                 INTEGER UNSIGNED   NOT NULL,
  entity_qtype_id         INTEGER UNSIGNED   NOT NULL,
  authorization_id        INTEGER UNSIGNED   NOT NULL,
  authorized_flag         BIT(1)   NOT NULL   DEFAULT false,
  created_by              INTEGER UNSIGNED   NULL,
  creation_date           DATETIME   NULL   DEFAULT NULL,
  modified_by             INTEGER UNSIGNED   NULL,
  modified_date           TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( role_entity_built_in_id ),
    INDEX role_entity_qtype_built_in_authorization_fkindex1 ( role_id ),
    INDEX role_entity_qtype_built_in_authorization_fkindex2 ( entity_qtype_id ),
    INDEX role_entity_qtype_built_in_authorization_fkindex3 ( authorization_id ),
    UNIQUE (role_id,entity_qtype_id,authorization_id ),
    INDEX role_entity_qtype_built_in_authorization_fkindex4 ( created_by ),
    INDEX role_entity_qtype_built_in_authorization_fkindex5 ( modified_by ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE role_entity_qtype_custom_field_authorization (
  role_entity_qtype_custom_field_authorization_id INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  role_id                                         INTEGER UNSIGNED   NOT NULL,
  entity_qtype_custom_field_id                    INTEGER UNSIGNED   NOT NULL,
  authorization_id                                INTEGER UNSIGNED   NOT NULL,
  authorized_flag                                 BIT(1)   NOT NULL   DEFAULT false,
  created_by                                      INTEGER UNSIGNED   NULL,
  creation_date                                   DATETIME   NULL   DEFAULT NULL,
  modified_by                                     INTEGER UNSIGNED   NULL,
  modified_date                                   TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( role_entity_qtype_custom_field_authorization_id ),
    INDEX role_entity_qtype_custom_field_authorization_fkindex1 ( role_id ),
    INDEX role_entity_qtype_custom_field_authorization_fkindex2 ( entity_qtype_custom_field_id ),
    INDEX role_entity_qtype_custom_field_authorization_fkindex3 ( authorization_id ),
    INDEX role_entity_qtype_custom_field_authorization_fkindex4 ( created_by ),
    INDEX role_entity_qtype_custom_field_authorization_fkindex5 ( modified_by ),
    UNIQUE (role_id,entity_qtype_custom_field_id,authorization_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE asset_transaction_checkout (
  asset_transaction_checkout_id      INTEGER UNSIGNED   NOT NULL   AUTO_INCREMENT,
  asset_transaction_id  INTEGER UNSIGNED   NOT NULL,
  to_contact_id   INTEGER UNSIGNED   NULL,
  to_user_id   INTEGER UNSIGNED   NULL,
  due_date        DATETIME   NULL,
  created_by      INTEGER UNSIGNED   NULL,
  creation_date   DATETIME   NULL,
  modified_by     INTEGER UNSIGNED   NULL,
  modified_date   TIMESTAMP ON UPDATE CURRENT_TIMESTAMP   NULL   DEFAULT NULL,
    PRIMARY KEY ( asset_transaction_checkout_id ),
    INDEX asset_transaction_checkout_fkindex1 ( to_contact_id ),
    INDEX asset_transaction_checkout_fkindex2 ( to_user_id ),
    INDEX asset_transaction_checkout_fkindex3 ( created_by ),
    INDEX asset_transaction_checkout_fkindex4 ( modified_by ),
    INDEX asset_transaction_checkout_fkindex5 ( asset_transaction_id ),
    UNIQUE (asset_transaction_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE address_custom_field_helper (
  address_id INTEGER UNSIGNED NOT NULL,
        PRIMARY KEY ( address_id),
        INDEX address_custom_field_helper_fkindex1 ( address_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE asset_custom_field_helper (
  asset_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( asset_id),
  	INDEX asset_custom_field_helper_fkindex1 ( asset_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE inventory_model_custom_field_helper (
  inventory_model_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( inventory_model_id),
  	INDEX inventory_model_custom_field_helper_fkindex1 ( inventory_model_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE asset_model_custom_field_helper (
  asset_model_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( asset_model_id),
  	INDEX asset_model_custom_field_helper_fkindex1 ( asset_model_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE manufacturer_custom_field_helper (
  manufacturer_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( manufacturer_id),
  	INDEX manufacturer_custom_field_helper_fkindex1 ( manufacturer_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE category_custom_field_helper (
  category_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( category_id),
  	INDEX category_custom_field_helper_fkindex1 ( category_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE company_custom_field_helper (
  company_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( company_id),
  	INDEX company_custom_field_helper_fkindex1 ( company_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE contact_custom_field_helper (
  contact_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( contact_id),
  	INDEX contact_custom_field_helper_fkindex1 ( contact_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE shipment_custom_field_helper (
  shipment_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( shipment_id),
  	INDEX shipment_custom_field_helper_fkindex1 ( shipment_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE receipt_custom_field_helper (
  receipt_id INTEGER UNSIGNED NOT NULL,
  	PRIMARY KEY ( receipt_id),
  	INDEX receipt_custom_field_helper_fkindex1 ( receipt_id ))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE  depreciation_class(
   depreciation_class_id INTEGER UNSIGNED NOT NULL AUTO_INCREMENT,
   depreciation_method_qtype_id INTEGER UNSIGNED NOT NULL,
   short_description VARCHAR(255)   NOT NULL,
   life INTEGER UNSIGNED   NULL,
   PRIMARY KEY (depreciation_class_id),
   INDEX depreciation_class_fkindex1 ( depreciation_class_id ),
   UNIQUE (short_description),
   INDEX depreciation_class_fkindex2 ( depreciation_method_qtype_id )
)
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE depreciation_method_qtype(
  depreciation_method_qtype_id  INTEGER UNSIGNED NOT NULL AUTO_INCREMENT,
  short_description  VARCHAR(255)   NOT NULL,
  PRIMARY KEY (depreciation_method_qtype_id),
  INDEX depreciation_method_qtype_fkindex1 (depreciation_method_qtype_id),
  INDEX `short_description_UNIQUE` (`short_description` ASC),
  UNIQUE (short_description))
ENGINE = INNODB DEFAULT CHARSET=utf8;

CREATE TABLE IF NOT EXISTS `email_queue` (
  `email_queue_id` INTEGER UNSIGNED NOT NULL AUTO_INCREMENT,
  `to_address` VARCHAR(128) DEFAULT NULL,
  `from_address` VARCHAR(128) DEFAULT NULL,
  `subject` VARCHAR(255) DEFAULT NULL,
  `body` TEXT,
  `html` TEXT,
  `error_flag` BOOLEAN DEFAULT NULL,
  `error_message` VARCHAR(255) DEFAULT NULL,
  PRIMARY KEY (`email_queue_id`)
) ENGINE=INNODB DEFAULT CHARSET=utf8;

ALTER TABLE depreciation_class
  ADD CONSTRAINT FOREIGN KEY(depreciation_method_qtype_id) references depreciation_method_qtype (
    depreciation_method_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_model
  ADD CONSTRAINT FOREIGN KEY( category_id) references category (
    category_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_model
  ADD CONSTRAINT FOREIGN KEY( manufacturer_id) references manufacturer (
    manufacturer_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_model
  ADD CONSTRAINT FOREIGN KEY(depreciation_class_id) references depreciation_class (
    depreciation_class_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset
  ADD CONSTRAINT FOREIGN KEY( asset_model_id) references asset_model (
    asset_model_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset
  ADD CONSTRAINT FOREIGN KEY( location_id) references location (
    location_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset
	ADD CONSTRAINT FOREIGN KEY ( `parent_asset_id` ) REFERENCES `asset` ( `asset_id` )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_model
  ADD CONSTRAINT FOREIGN KEY( category_id) references category (
    category_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_model
  ADD CONSTRAINT FOREIGN KEY( manufacturer_id) references manufacturer (
    manufacturer_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_location
  ADD CONSTRAINT FOREIGN KEY( location_id) references location (
    location_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE inventory_location
  ADD CONSTRAINT FOREIGN KEY( inventory_model_id) references inventory_model (
    inventory_model_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE user_account
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE user_account
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_model
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_model
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE location
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE location
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE manufacturer
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE manufacturer
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_model
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_model
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE category
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE category
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_transaction
  ADD CONSTRAINT FOREIGN KEY( transaction_id) references transaction (
    transaction_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE transaction
  ADD CONSTRAINT FOREIGN KEY( transaction_type_id) references transaction_type (
    transaction_type_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_transaction
  ADD CONSTRAINT FOREIGN KEY( source_location_id) references location (
    location_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_transaction
  ADD CONSTRAINT FOREIGN KEY( destination_location_id) references location (
    location_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE transaction
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE transaction
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( transaction_id) references transaction (
    transaction_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( source_location_id) references location (
    location_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( destination_location_id) references location (
    location_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE custom_field
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE custom_field
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE custom_field_value
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE custom_field_value
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE custom_field
  ADD CONSTRAINT FOREIGN KEY( custom_field_qtype_id) references custom_field_qtype (
    custom_field_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE custom_field_value
  ADD CONSTRAINT FOREIGN KEY( custom_field_id) references custom_field (
    custom_field_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE entity_qtype_custom_field
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_id) references entity_qtype (
    entity_qtype_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE entity_qtype_custom_field
  ADD CONSTRAINT FOREIGN KEY( custom_field_id) references custom_field (
    custom_field_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_custom_field_asset_model
  ADD CONSTRAINT FOREIGN KEY( asset_model_id) references asset_model (
   asset_model_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_custom_field_asset_model
  ADD CONSTRAINT FOREIGN KEY( custom_field_id) references custom_field (
    custom_field_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( asset_id) references asset (
    asset_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE address
  ADD CONSTRAINT FOREIGN KEY( company_id) references company (
    company_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE address
  ADD CONSTRAINT FOREIGN KEY( country_id) references country (
    country_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE state_province
  ADD CONSTRAINT FOREIGN KEY( country_id) references country (
    country_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE address
  ADD CONSTRAINT FOREIGN KEY( state_province_id) references state_province (
    state_province_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_transaction
  ADD CONSTRAINT FOREIGN KEY( inventory_location_id) references inventory_location (
    inventory_location_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE transaction
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_id) references entity_qtype (
    entity_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE company
  ADD CONSTRAINT FOREIGN KEY( address_id) references address (
    address_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE company
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE company
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE contact
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE contact
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE address
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE address
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE contact
  ADD CONSTRAINT FOREIGN KEY( address_id) references address (
    address_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( transaction_id) references transaction (
    transaction_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( from_address_id) references address (
    address_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( to_address_id) references address (
    address_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( to_company_id) references company (
    company_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( courier_id) references courier (
    courier_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( from_company_id) references company (
    company_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( from_contact_id) references contact (
    contact_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( to_contact_id) references contact (
    contact_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( to_address_id) references address (
    address_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE receipt
  ADD CONSTRAINT FOREIGN KEY( transaction_id) references transaction (
    transaction_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_transaction_checkout
  ADD CONSTRAINT FOREIGN KEY( to_contact_id) references contact (
    contact_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction_checkout
  ADD CONSTRAINT FOREIGN KEY( to_user_id) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction_checkout
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction_checkout
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction_checkout
  ADD CONSTRAINT FOREIGN KEY( asset_transaction_id) references asset_transaction (
    asset_transaction_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE contact
  ADD CONSTRAINT FOREIGN KEY( company_id) references company (
    company_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( from_contact_id) references contact (
    contact_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( to_contact_id) references contact (
    contact_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE user_account
  ADD CONSTRAINT FOREIGN KEY( role_id) references role (
    role_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_module
  ADD CONSTRAINT FOREIGN KEY( role_id) references role (
    role_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE role_module
  ADD CONSTRAINT FOREIGN KEY( module_id) references module (
    module_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_module_authorization
  ADD CONSTRAINT FOREIGN KEY( role_module_id) references role_module (
    role_module_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE role_module_authorization
  ADD CONSTRAINT FOREIGN KEY( authorization_id) references authorization (
    authorization_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_module_authorization
  ADD CONSTRAINT FOREIGN KEY( authorization_level_id) references authorization_level (
    authorization_level_id
  )
ON Delete NO ACTION ON Update NO ACTION;

alter table role_transaction_type_authorization
	add constraint
	foreign key (
		created_by
	) references user_account (
		user_account_id
	)
	ON Delete NO ACTION ON Update NO ACTION;

alter table role_transaction_type_authorization
	add constraint
	foreign key (
		modified_by
	) references user_account (
		user_account_id
	)
	ON Delete NO ACTION ON Update NO ACTION;

alter table role_transaction_type_authorization
	add constraint
	foreign key (
		authorization_level_id
	) references authorization_level (
		authorization_level_id
	)
	ON Delete NO ACTION ON Update NO ACTION;

alter table role_transaction_type_authorization
	add constraint
	foreign key (
		transaction_type_id
	) references transaction_type (
		transaction_type_id
	)
	ON Delete NO ACTION ON Update NO ACTION;

alter table role_transaction_type_authorization
	add constraint
	foreign key (
		role_id
	) references role (
		role_id
	)
	ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE role
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shortcut
  ADD CONSTRAINT FOREIGN KEY( module_id) references module (
    module_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shortcut
  ADD CONSTRAINT FOREIGN KEY( authorization_id) references authorization (
    authorization_id
  )
ON Delete NO ACTION ON Update NO ACTION;

alter table shortcut
	add constraint
	foreign key (
		transaction_type_id
	) references transaction_type (
		transaction_type_id
	)
	ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE inventory_location
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE inventory_location
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE inventory_transaction
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE inventory_transaction
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE role_module
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
)
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE role_module
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE role_module_authorization
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE role_module_authorization
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE custom_field
  ADD CONSTRAINT FOREIGN KEY( default_custom_field_value_id) references custom_field_value (
    custom_field_value_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( new_asset_id) references asset (
    asset_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE shipment
  ADD CONSTRAINT FOREIGN KEY( from_company_id) references company (
    company_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE datagrid_column_preference
  ADD CONSTRAINT FOREIGN KEY( datagrid_id) references datagrid (
    datagrid_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE datagrid_column_preference
  ADD CONSTRAINT FOREIGN KEY( user_account_id) references user_account (
    user_account_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE notification
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE notification
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE notification_user_account
  ADD CONSTRAINT FOREIGN KEY( notification_id) references notification (
    notification_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE notification_user_account
  ADD CONSTRAINT FOREIGN KEY( user_account_id) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE attachment
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_id) references entity_qtype (
    entity_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE attachment
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE asset_transaction
  ADD CONSTRAINT FOREIGN KEY( parent_asset_transaction_id) references asset_transaction (
    asset_transaction_id
  )
ON Delete SET NULL ON Update NO ACTION;

ALTER TABLE audit
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_id) references entity_qtype (
    entity_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE audit
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE audit
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE audit_scan
  ADD CONSTRAINT FOREIGN KEY( audit_id) references audit (
    audit_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE audit_scan
  ADD CONSTRAINT FOREIGN KEY( location_id) references location (
    location_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_built_in_authorization
  ADD CONSTRAINT FOREIGN KEY( role_id) references role (
    role_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_built_in_authorization
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_id) references entity_qtype (
    entity_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_built_in_authorization
  ADD CONSTRAINT FOREIGN KEY( authorization_id) references authorization (
    authorization_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_custom_field_authorization
  ADD CONSTRAINT FOREIGN KEY( role_id) references role (
    role_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_custom_field_authorization
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_custom_field_id) references entity_qtype_custom_field (
    entity_qtype_custom_field_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_custom_field_authorization
  ADD CONSTRAINT FOREIGN KEY( authorization_id) references authorization (
    authorization_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_custom_field_authorization
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_custom_field_authorization
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE shortcut
  ADD CONSTRAINT FOREIGN KEY( entity_qtype_id) references entity_qtype (
    entity_qtype_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE address_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( address_id) references address (
    address_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( asset_id) references asset (
    asset_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE inventory_model_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( inventory_model_id) references  inventory_model (
    inventory_model_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE asset_model_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( asset_model_id) references asset_model (
    asset_model_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE manufacturer_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( manufacturer_id) references manufacturer (
    manufacturer_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE category_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( category_id) references category (
    category_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE company_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( company_id) references company (
    company_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE contact_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( contact_id) references contact (
    contact_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE shipment_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( shipment_id) references shipment (
    shipment_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE receipt_custom_field_helper
  ADD CONSTRAINT FOREIGN KEY( receipt_id) references receipt (
    receipt_id
  )
ON Delete CASCADE ON Update NO ACTION;

ALTER TABLE role_entity_qtype_built_in_authorization
  ADD CONSTRAINT FOREIGN KEY( created_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;

ALTER TABLE role_entity_qtype_built_in_authorization
  ADD CONSTRAINT FOREIGN KEY( modified_by) references user_account (
    user_account_id
  )
ON Delete NO ACTION ON Update NO ACTION;
